<?php
/**
 * Plugin Name: PeepSo Tools: BuddyBoss Migrator
 * Description: Migration plugin from BuddyBoss to PeepSo
 * Tags: peepso, buddyboss, integration
 * Author: PeepSo
 * Version: 8.0.1.0
 * Author URI: https://www.PeepSo.com
 * Copyright: (c) PeepSo, Inc. All Rights Reserved.
 * License: GPLv2 or later
 * License URI: http://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: peepsobuddyboss
 * Domain Path: /language
 *
 * We are Open Source. You can redistribute and/or modify this software under the terms of the GNU General Public License (version 2 or later)
 * as published by the Free Software Foundation. See the GNU General Public License or the LICENSE file for more details.
 * This software is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY.
 */

class PeepsoMigrateBB {

	private static $_instance = NULL;

	const DEBUG = TRUE;
	const MODULE_ID = 0;
    const PLUGIN_VERSION = '8.0.1.0';
    const PLUGIN_RELEASE = ''; //ALPHA1, RC1 etc, '' for STABLE
	const PLUGIN_NAME = 'PeepsoMigrateBB';
	const PLUGIN_SLUG = 'peepsomigrate_';

	public function __construct() {
		/** VERSION LOCKED hooks **/
        if(self::ready()) {
			add_action('peepso_init', array(&$this, 'init'));
		}

		if (is_admin()) {
			add_action('admin_init', array(&$this, 'peepso_check'));
		}

		add_action('init', array(&$this, 'load_textdomain'));
		add_filter('peepso_all_plugins', array($this, 'filter_all_plugins'));

		register_activation_hook(__FILE__, array(&$this, 'activate'));
	}

    private static function ready_thirdparty() {
        $result = TRUE;

        if(!class_exists('BuddyPress')) {
            $result = FALSE;
        }

        return $result;
    }

	private static function ready() {
        if (class_exists('PeepSo') && PeepSo::ready(self::PLUGIN_VERSION) && self::ready_thirdparty()) {
            return TRUE;
        }
    }

	/**
	 * Return singleton instance of plugin
	 */
	public static function get_instance() {
		if (NULL === self::$_instance) {
			self::$_instance = new self();
		}

		return (self::$_instance);
	}

	/**
	 * Loads the translation file for the PeepSo plugin
	 */
	public function load_textdomain() {
		$path = plugin_basename(dirname(__FILE__)) . DIRECTORY_SEPARATOR . 'language';
		load_plugin_textdomain('peepsobuddyboss', false, $path);
	}

	public function init() {
		require_once 'classes/migrate.php';
		PeepSoTemplate::add_template_directory(plugin_dir_path(__FILE__));

		if (is_admin()) {
			add_action('admin_menu', array(&$this, 'admin_menu'), 9);
			add_action('admin_init', array(&$this, 'peepso_check'));

			$get = array_map('sanitize_text_field', $_GET);

			if ((!empty($_SERVER['HTTP_X_REQUESTED_WITH']) || isset($get['debug'])) && isset($get['page']) && $get['page'] == 'pm-dashboard') {
				add_action('admin_init', function() {
					$this->migrate();
					die();
				});
			} else if (isset($get['page']) && $get['page'] == 'pm-dashboard') {
				wp_register_style('peepsobuddyboss', PeepsoMigrateBB::get_asset('css/peepsobuddyboss.css'), NULL, PeepsoMigrateBB::PLUGIN_VERSION, 'all');
				wp_enqueue_style('peepsobuddyboss');
			}
		}
	}

	/**
	 * Plugin activation
	 * Check PeepSo
	 * @return bool
	 */
	public function activate() {
		if (!$this->peepso_check()) {
			return (FALSE);
		}

		return (TRUE);
	}

	/**
	 * Check if PeepSo class is present (ie the PeepSo plugin is installed and activated)
	 * If there is no PeepSo, immediately disable the plugin and display a warning
	 * Run license and new version checks against PeepSo.com
	 * @return bool
	 */
	public function peepso_check() {
		if (!class_exists('PeepSo')) {
			add_action('admin_notices', array(&$this, 'peepso_disabled_notice'));
			deactivate_plugins(plugin_basename(__FILE__));
			unset($_GET['activate']);
			return (FALSE);
		}

		if (!class_exists('BuddyPress')) {
			add_action('admin_notices', array(&$this, 'bp_disabled_notice'));
			deactivate_plugins(plugin_basename(__FILE__));
			unset($_GET['activate']);
			return (FALSE);
			
		}

		$bb_plugin_file = WP_PLUGIN_DIR . DIRECTORY_SEPARATOR . 'buddyboss-platform' . DIRECTORY_SEPARATOR . 'bp-loader.php';
		$bb_details = get_plugin_data($bb_plugin_file, true, true);

		if (version_compare($bb_details['Version'], '2.0', '<')) {
			add_action('admin_notices', array(&$this, 'version_notice'));
			deactivate_plugins(plugin_basename(__FILE__));
			unset($_GET['activate']);
			return (FALSE);
		}

		ob_start();
		PeepSo::plugins_version_notice();
		$version_notice = ob_get_clean();
		
		if (!empty($version_notice)) {
			add_action('admin_notices', array(&$this, 'version_mismatch_notice'));
			define('PEEPSO_VERSION_MISMATCH', TRUE);
		}

		return (TRUE);
	}

	/**
	 * Display a message about PeepSo not present
	 */
	public function peepso_disabled_notice() {
		?>
		<div class="error">
			<strong>
				<?php echo sprintf(__('The %s plugin requires the PeepSo plugin to be installed and activated.', 'peepsobuddyboss'), self::PLUGIN_NAME); ?>
			</strong>
		</div>
		<?php
	}

	/**
	 * Display a message about BP not present
	 */
	public function bp_disabled_notice() {
		?>
		<div class="error">
			<strong>
				<?php echo sprintf(__('The %s plugin requires the BuddyBoss plugin to be installed and activated.', 'peepsobuddyboss'), self::PLUGIN_NAME); ?>
			</strong>
		</div>
		<?php
	}

	/**
	 * Display a message about BP not present
	 */
	public function version_notice() {
		?>
		<div class="error">
			<strong>
				<?php echo __('BuddyBoss plugin version should be greater than 2.5', 'peepsobuddyboss'); ?>
			</strong>
		</div>
		<?php
	}

	/**
	 * Display a message about PeepSo version mismatch
	 */
	public function version_mismatch_notice() {
		?>
		<div class="error">
			<strong>
				<?php echo __('Please update your PeepSo plugins to the latest version to perform migration', 'peepsobuddyboss'); ?>
			</strong>
		</div>
		<?php
	}

	/**
	 * Hooks into PeepSo for compatibility checks
	 * @param $plugins
	 * @return mixed
	 */
	public function filter_all_plugins($plugins) {
		$plugins[plugin_basename(__FILE__)] = get_class($this);
		return $plugins;
	}

	public function admin_menu() {
		$dashboard_hookname = add_menu_page(__('PeepsoMigrateBB', 'peepsobuddyboss'), __('PeepsoMigrateBB', 'peepsobuddyboss'), 'manage_options', 'pm-dashboard', array(&$this, 'pm_dashboard'), PeepsoMigrateBB::get_asset('images/logo-icon_20x20.png'), 5);
	}

	public function pm_dashboard() {
		$post = array_map('sanitize_text_field', $_POST);

		if (isset($post['start_peepso_migrate'])) {
			global $wpdb;

			if (isset($post['gender_field'])) {
				delete_transient('peepso_migrate_gender_field');
				delete_transient('peepso_migrate_gender_field_male');
				delete_transient('peepso_migrate_gender_field_female');

				set_transient('peepso_migrate_gender_field', $post['gender_field']);

				$gender_field_male = isset($post['gender_field_male_' . $post['gender_field']]) ? $post['gender_field_male_' . $post['gender_field']] : '';
				$gender_field_female = isset($post['gender_field_female_' . $post['gender_field']]) ? $post['gender_field_female_' . $post['gender_field']] : '';

				if (!empty($gender_field_male) && !empty($gender_field_female)) {

					set_transient('peepso_migrate_gender_field_male', $post['gender_field_male_' . $post['gender_field']]);
					set_transient('peepso_migrate_gender_field_female', $post['gender_field_female_' . $post['gender_field']]);
				}

				$result_gender_select = $wpdb->get_results($wpdb->prepare("SELECT id, name FROM " . $wpdb->prefix . "bp_xprofile_fields WHERE type NOT IN('wp-biography', 'wp-textbox') AND parent_id = %d AND id NOT IN (%d, %d) ORDER BY id ASC", $post['gender_field'], $gender_field_male, $gender_field_female));

				for ($i = 1; $i <= count($result_gender_select); $i++) {

					// always delete transient to prevent duplicate value
					delete_transient('peepso_migrate_gender_field_custom' . ($i + 2));

					if (isset($post['gender_field_custom' . ($i + 2) . '_' . $post['gender_field']])) {
						set_transient('peepso_migrate_gender_field_custom' . ($i + 2), $post['gender_field_custom' . ($i + 2) . '_' . $post['gender_field']]);
					}
				}
			}

			if (isset($post['birthdate_field'])) {
				delete_transient('peepso_migrate_birthdate_field');
				set_transient('peepso_migrate_birthdate_field', $post['birthdate_field']);
			}

			$active_components = apply_filters('bp_active_components', bp_get_option('bp-active-components'));

			$data = array(
				'next_url' => PeepsoMigrateBB::url('delete')
			);

			$sql = "SELECT count(*) as total FROM " . $wpdb->prefix . "users";

			$data['total_profiles'] = $data['total_avatars'] = $wpdb->get_row($sql)->total;

			if (self::plugin_check('friend')) {
				$sql = "SELECT count(*) as total FROM " . $wpdb->prefix . "bp_friends";
				$data['total_friends'] = $wpdb->get_row($sql)->total;
			}

			if (self::plugin_check('activity')) {
				$sql = "SELECT count(*) as total FROM " . $wpdb->prefix . "bp_activity where `component` not in ('members', 'groups', 'friends', 'profile')";
				$data['total_activities'] = $wpdb->get_row($sql)->total;

				if(class_exists('PeepSoSharePhotos')) {
					$sql = "SELECT count(*) as total FROM " . $wpdb->prefix . "bp_media_albums where `group_id` = 0";
					$data['total_album'] = $wpdb->get_row($sql)->total;
				}
			}

			if (self::plugin_check('notification')) {
				$sql = "SELECT count(*) as total FROM " . $wpdb->prefix . "bp_notifications bpn, " . $wpdb->prefix . "bp_activity bpa where bpn.`item_id`= bpa.`id` AND bpn.`component_name` = 'activity'";
				$data['total_notifications'] = $wpdb->get_row($sql)->total;
			}

			if (self::plugin_check('message')) {
				$sql = "SELECT count(*) as total FROM " . $wpdb->prefix . "bp_messages_messages";
				$data['total_messages'] = $wpdb->get_row($sql)->total;
			}

			if (self::plugin_check('profile_field')) {
				$gender_field = get_transient('peepso_migrate_gender_field');
				$gender_field_male = get_transient('peepso_migrate_gender_field_male');
				$gender_field_female = get_transient('peepso_migrate_gender_field_female');

				$birthdate_field = get_transient('peepso_migrate_birthdate_field');

				// get all profile fields, exclude name, gender, and birthdate
				if (!empty($gender_field) && !empty($gender_field_male) && !empty($gender_field_female)) {
					$sql = $wpdb->prepare("SELECT count(*) as total FROM " . $wpdb->prefix . "bp_xprofile_fields WHERE type NOT IN('wp-biography', 'wp-textbox', 'socialnetworks') AND parent_id = 0 AND id NOT IN (%d, %d, %d, 1, 2) ORDER BY id, group_id", 1, intval($gender_field), intval($birthdate_field));
				} else {
					$sql = $wpdb->prepare("SELECT count(*) as total FROM " . $wpdb->prefix . "bp_xprofile_fields WHERE type NOT IN('wp-biography', 'wp-textbox', 'socialnetworks') AND parent_id = 0 AND id NOT IN (%d, %d, 1, 2) ORDER BY id, group_id", 1, intval($birthdate_field));
				}
				$data['total_profile_fields'] = $wpdb->get_row($sql)->total;
			}

			if (self::plugin_check('group')) {
				$sql = "SELECT count(*) as total FROM " . $wpdb->prefix . "bp_groups";
				$data['total_groups'] = $data['total_group_avatars'] = $wpdb->get_row($sql)->total;

				$sql = "SELECT count(*) as total FROM " . $wpdb->prefix . "bp_groups_members";
				$data['total_group_members'] = $wpdb->get_row($sql)->total;


				$sql = "SELECT count(*) as total FROM " . $wpdb->prefix . "bp_activity where `component` = 'groups' or `type` = 'activity_comment'";
				$data['total_group_activity'] = $wpdb->get_row($sql)->total;

				if(class_exists('PeepSoSharePhotos')) {
					$sql = "SELECT count(*) as total FROM " . $wpdb->prefix . "bp_media_albums where `group_id` > 0";
					$data['total_group_album'] = $wpdb->get_row($sql)->total;
				}
			}

			echo esc_html(PeepSoTemplate::exec_template('migrate', 'progress', $data));
		} else {
			echo esc_html(PeepSoTemplate::exec_template('migrate', 'dashboard'));
		}
	}

	public static function get_asset($ref) {
		$ret = plugin_dir_url(__FILE__) . 'assets/' . $ref;
		return ($ret);
	}

	public function migrate() {
		$functions = [
			'delete'                  => ['deletePeepSoContent', 0],
			'profile'                 => ['startMigrateProfiles', 1],
			'avatar'                  => ['startMigrateAvatars', 2],
			'friend'                  => ['startMigrateFriends', 3],
			'album'                   => ['startMigrateAlbum', 4],
			'activity'                => ['startMigrateActivity', 5],
			'notification'            => ['startMigrateNotifications', 6],
			'message'                 => ['startMigrateMessages', 7],
			'profile_field'           => ['startMigrateProfileFields', 8],
			'profile_field_value'     => ['startMigrateProfileFieldValue', 8],
			'profile_field_user_data' => ['startMigrateProfileFieldUserData', 8],
			'group'                   => ['startMigrateGroups', 9],
			'group_member'            => ['startMigrateGroupMembers', 10],
			'group_avatar'            => ['startMigrateGroupAvatars', 11],
			'group_album'             => ['startMigrateGroupAlbum', 12],
			'group_activity'          => ['startMigrateGroupActivity', 13],
			'unpublish'               => ['unpublishBuddyBoss', 14],
		];

		$action = sanitize_text_field($_GET['action']);


		if (defined('PEEPSO_MIGRATE_SKIP_' . strtoupper($action))) {
			$keys = array_keys($functions);
    		$index = array_search($action, $keys);

			$response = [
				'class' => $action,
				'counter' => 0,
			];

			if ($index === FALSE || $index === count($keys) - 1) {
				$response['finished'] = 1;
			} else {
				$response['url'] = PeepsoMigrateBB::url($keys[$index + 1]);
			}

		} else if (array_key_exists($action, $functions)) {
			$migrate = new PeepsoMigrateBBAdminMigrate();
			$response = $migrate->{$functions[$action][0]}();
		}

		$response['general'] = $functions[$action][1];

		echo json_encode($response);
		die();
	}

	public static function url($action) {
		return admin_url('admin.php?page=pm-dashboard&action=' . $action);
	}

	public static function plugin_check($plugin) {

		$active_components = apply_filters('bp_active_components', bp_get_option('bp-active-components'));

		switch ($plugin) {
			case 'activity' :
				if (isset($active_components['activity']) && 1 == $active_components['activity']) {
					return TRUE;
				}
			case 'notification' :
				if (isset($active_components['notifications']) && 1 == $active_components['notifications']) {
					return TRUE;
				}
			case 'friend' :
				if (class_exists('PeepSoFriendsPlugin') && isset($active_components['friends']) && 1 == $active_components['friends']) {
					return TRUE;
				}
				break;
			case 'message' :
				if (class_exists('PeepSoMessagesPlugin') && isset($active_components['messages']) && 1 == $active_components['messages']) {
					return TRUE;
				}
				break;
			case 'profile_field' :
				if (isset($active_components['xprofile']) && 1 == $active_components['xprofile']) {
					return TRUE;
				}
				break;
			case 'group' :
				if (class_exists('PeepSoGroupsPlugin') && isset($active_components['groups']) && 1 == $active_components['groups']) {
					return TRUE;
				}
				break;
		}

		return FALSE;
	}

}

defined('WPINC') || die;
$peepso_migrate = new PeepsoMigrateBB();
?>
