<?php
/**
 * Plugin Name: PeepSo Tools: BuddyPress Migrator
 * Description: Migration plugin from BuddyPress to PeepSo
 * Tags: peepso, wpadverts, integration
 * Author: PeepSo
 * Version: 7.1.0.0
 * Author URI: https://www.PeepSo.com
 * Copyright: (c) 2015 PeepSo, Inc. All Rights Reserved.
 * License: GPLv2 or later
 * License URI: http://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: peepsomigrate
 * Domain Path: /language
 *
 * We are Open Source. You can redistribute and/or modify this software under the terms of the GNU General Public License (version 2 or later)
 * as published by the Free Software Foundation. See the GNU General Public License or the LICENSE file for more details.
 * This software is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY.
 */

class PeepsoMigrate {

	private static $_instance = NULL;

	const DEBUG = TRUE;
	const MODULE_ID = 0;
	const PLUGIN_VERSION = '7.1.0.0';
	const PLUGIN_RELEASE = ''; //ALPHA1, BETA10, RC1, '' for STABLE
	const PLUGIN_NAME = 'PeepsoMigrate';
	const PLUGIN_SLUG = 'peepsomigrate_';

	public function __construct() {
		/** VERSION LOCKED hooks **/
        if(self::ready()) {
			add_action('peepso_init', array(&$this, 'init'));
		}

		if (is_admin()) {
			add_action('admin_init', array(&$this, 'peepso_check'));
		}

		add_action('init', array(&$this, 'load_textdomain'));
		add_filter('peepso_all_plugins', array($this, 'filter_all_plugins'));

		register_activation_hook(__FILE__, array(&$this, 'activate'));
	}

    private static function ready_thirdparty() {
        $result = TRUE;

        if(!class_exists('BuddyPress')) {
            $result = FALSE;
        }

        return $result;
    }

	private static function ready() {
        if (class_exists('PeepSo') && PeepSo::ready(self::PLUGIN_VERSION) && self::ready_thirdparty()) {
            return TRUE;
        }
    }

	/**
	 * Return singleton instance of plugin
	 */
	public static function get_instance() {
		if (NULL === self::$_instance) {
			self::$_instance = new self();
		}

		return (self::$_instance);
	}

	/**
	 * Loads the translation file for the PeepSo plugin
	 */
	public function load_textdomain() {
		$path = plugin_basename(dirname(__FILE__)) . DIRECTORY_SEPARATOR . 'language';
		load_plugin_textdomain('peepsomigrate', FALSE, $path);
	}

	public function init() {
//		PeepSo::add_autoload_directory(dirname(__FILE__) . DIRECTORY_SEPARATOR . 'classes' . DIRECTORY_SEPARATOR);
		require_once 'classes/migrate.php';
		PeepSoTemplate::add_template_directory(plugin_dir_path(__FILE__));

		if (is_admin()) {
			add_action('admin_menu', array(&$this, 'admin_menu'), 9);
			add_action('admin_init', array(&$this, 'peepso_check'));

			if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && isset($_GET['page']) && $_GET['page'] == 'pm-dashboard') {
				add_action('admin_init', array($this, 'migrate'), 99);
			} else if (isset($_GET['page']) && $_GET['page'] == 'pm-dashboard') {
				wp_register_style('peepsomigrate', PeepsoMigrate::get_asset('css/peepsomigrate.css'), NULL, peepsomigrate::PLUGIN_VERSION, 'all');
				wp_enqueue_style('peepsomigrate');
			}
		}
	}

	/**
	 * Plugin activation
	 * Check PeepSo
	 * @return bool
	 */
	public function activate() {
		if (!$this->peepso_check()) {
			return (FALSE);
		}

		return (TRUE);
	}

	/**
	 * Check if PeepSo class is present (ie the PeepSo plugin is installed and activated)
	 * If there is no PeepSo, immediately disable the plugin and display a warning
	 * Run license and new version checks against PeepSo.com
	 * @return bool
	 */
	public function peepso_check() {
		if (!class_exists('PeepSo')) {
			add_action('admin_notices', array(&$this, 'peepso_disabled_notice'));
			deactivate_plugins(plugin_basename(__FILE__));
			unset($_GET['activate']);
			return (FALSE);
		}

		if (!class_exists('BuddyPress')) {
			add_action('admin_notices', array(&$this, 'bp_disabled_notice'));
			deactivate_plugins(plugin_basename(__FILE__));
			unset($_GET['activate']);
			return (FALSE);
			
		}

		$bp_plugin_file = WP_PLUGIN_DIR . DIRECTORY_SEPARATOR . 'buddypress' . DIRECTORY_SEPARATOR . 'bp-loader.php';
		$bp_details = get_plugin_data($bp_plugin_file, true, true);

		if (version_compare($bp_details['Version'], '2.5', '<')) {
			add_action('admin_notices', array(&$this, 'version_notice'));
			deactivate_plugins(plugin_basename(__FILE__));
			unset($_GET['activate']);
			return (FALSE);
		}

		ob_start();
		PeepSo::plugins_version_notice();
		$version_notice = ob_get_clean();
		
		if (!empty($version_notice)) {
			add_action('admin_notices', array(&$this, 'version_mismatch_notice'));
			define('PEEPSO_VERSION_MISMATCH', TRUE);
		}

		return (TRUE);
	}

	/**
	 * Display a message about PeepSo not present
	 */
	public function peepso_disabled_notice() {
		?>
		<div class="error">
			<strong>
				<?php echo sprintf(__('The %s plugin requires the PeepSo plugin to be installed and activated.', 'peepsomigrate'), self::PLUGIN_NAME); ?>
			</strong>
		</div>
		<?php
	}

	/**
	 * Display a message about BP not present
	 */
	public function bp_disabled_notice() {
		?>
		<div class="error">
			<strong>
				<?php echo sprintf(__('The %s plugin requires the BuddyPress plugin to be installed and activated.', 'peepsomigrate'), self::PLUGIN_NAME); ?>
			</strong>
		</div>
		<?php
	}

	/**
	 * Display a message about BP not present
	 */
	public function version_notice() {
		?>
		<div class="error">
			<strong>
				<?php echo __('BuddyPress plugin version should be greater than 2.5', 'peepsomigrate'); ?>
			</strong>
		</div>
		<?php
	}

	/**
	 * Display a message about PeepSo version mismatch
	 */
	public function version_mismatch_notice() {
		?>
		<div class="error">
			<strong>
				<?php echo __('Please update your PeepSo plugins to the latest version to perform migration', 'peepsomigrate'); ?>
			</strong>
		</div>
		<?php
	}

	/**
	 * Hooks into PeepSo for compatibility checks
	 * @param $plugins
	 * @return mixed
	 */
	public function filter_all_plugins($plugins) {
		$plugins[plugin_basename(__FILE__)] = get_class($this);
		return $plugins;
	}

	public function admin_menu() {
		$dashboard_hookname = add_menu_page(__('PeepsoMigrate', 'peepsomigrate'), __('PeepsoMigrate', 'peepsomigrate'), 'manage_options', 'pm-dashboard', array(&$this, 'pm_dashboard'), PeepsoMigrate::get_asset('images/logo-icon_20x20.png'), 5);
	}

	public function pm_dashboard() {
		if (isset($_POST['start_peepso_migrate'])) {
			global $wpdb;

			if (isset($_POST['gender_field'])) {
				delete_transient('peepso_migrate_gender_field');
				delete_transient('peepso_migrate_gender_field_male');
				delete_transient('peepso_migrate_gender_field_female');

				set_transient('peepso_migrate_gender_field', $_POST['gender_field']);

				$gender_field_male = isset($_POST['gender_field_male_' . $_POST['gender_field']]) ? $_POST['gender_field_male_' . $_POST['gender_field']] : '';
				$gender_field_female = isset($_POST['gender_field_female_' . $_POST['gender_field']]) ? $_POST['gender_field_female_' . $_POST['gender_field']] : '';

				if (!empty($gender_field_male) && !empty($gender_field_female)) {

					set_transient('peepso_migrate_gender_field_male', $_POST['gender_field_male_' . $_POST['gender_field']]);
					set_transient('peepso_migrate_gender_field_female', $_POST['gender_field_female_' . $_POST['gender_field']]);
				}

				$result_gender_select = $wpdb->get_results($wpdb->prepare("SELECT id, name FROM " . $wpdb->prefix . "bp_xprofile_fields WHERE type NOT IN('wp-biography', 'wp-textbox') AND parent_id = %d AND id NOT IN (%d, %d) ORDER BY id ASC", $_POST['gender_field'], $gender_field_male, $gender_field_female));

				for ($i = 1; $i <= count($result_gender_select); $i++) {

					// always delete transient to prevent duplicate value
					delete_transient('peepso_migrate_gender_field_custom' . ($i + 2));

					if (isset($_POST['gender_field_custom' . ($i + 2) . '_' . $_POST['gender_field']])) {
						set_transient('peepso_migrate_gender_field_custom' . ($i + 2), $_POST['gender_field_custom' . ($i + 2) . '_' . $_POST['gender_field']]);
					}
				}
			}

			if (isset($_POST['birthdate_field'])) {
				delete_transient('peepso_migrate_birthdate_field');
				set_transient('peepso_migrate_birthdate_field', $_POST['birthdate_field']);
			}

			$active_components = apply_filters('bp_active_components', bp_get_option('bp-active-components'));

			$data = array(
				'next_url' => PeepsoMigrate::url('delete')
			);

			$sql = "SELECT count(*) as total FROM " . $wpdb->prefix . "users";

			$data['total_profiles'] = $data['total_avatars'] = $wpdb->get_row($sql)->total;

			if (self::plugin_check('friend')) {
				$sql = "SELECT count(*) as total FROM " . $wpdb->prefix . "bp_friends";
				$data['total_friends'] = $wpdb->get_row($sql)->total;
			}

			if (self::plugin_check('activity')) {
				$sql = "SELECT count(*) as total FROM " . $wpdb->prefix . "bp_activity where `component` not in ('members', 'groups', 'friends')";
				$data['total_activities'] = $wpdb->get_row($sql)->total;
			}

			if (self::plugin_check('notification')) {
				$sql = "SELECT count(*) as total FROM " . $wpdb->prefix . "bp_notifications bpn, " . $wpdb->prefix . "bp_activity bpa where bpn.`item_id`= bpa.`id`";
				$data['total_notifications'] = $wpdb->get_row($sql)->total;
			}

			if (self::plugin_check('message')) {
				$sql = "SELECT count(*) as total FROM " . $wpdb->prefix . "bp_messages_messages";
				$data['total_messages'] = $wpdb->get_row($sql)->total;
			}

			if (self::plugin_check('profile_field')) {
				$gender_field = get_transient('peepso_migrate_gender_field');
				$gender_field_male = get_transient('peepso_migrate_gender_field_male');
				$gender_field_female = get_transient('peepso_migrate_gender_field_female');

				$birthdate_field = get_transient('peepso_migrate_birthdate_field');

				// get all profile fields, exclude name, gender, and birthdate
				if (!empty($gender_field) && !empty($gender_field_male) && !empty($gender_field_female)) {
					$sql = $wpdb->prepare("SELECT count(*) as total FROM " . $wpdb->prefix . "bp_xprofile_fields WHERE type NOT IN('wp-biography', 'wp-textbox') AND parent_id = 0 AND id NOT IN (%d, %d, %d) ORDER BY id, group_id", 1, intval($gender_field), intval($birthdate_field));
				} else {
					$sql = $wpdb->prepare("SELECT count(*) as total FROM " . $wpdb->prefix . "bp_xprofile_fields WHERE type NOT IN('wp-biography', 'wp-textbox') AND parent_id = 0 AND id NOT IN (%d, %d) ORDER BY id, group_id", 1, intval($birthdate_field));
				}
				$data['total_profile_fields'] = $wpdb->get_row($sql)->total;
			}

			if (self::plugin_check('group')) {
				$sql = "SELECT count(*) as total FROM " . $wpdb->prefix . "bp_groups";
				$data['total_groups'] = $data['total_group_avatars'] = $wpdb->get_row($sql)->total;

				$sql = "SELECT count(*) as total FROM " . $wpdb->prefix . "bp_groups_members";
				$data['total_group_members'] = $wpdb->get_row($sql)->total;


				$sql = "SELECT count(*) as total FROM " . $wpdb->prefix . "bp_activity where `component` = 'groups'";
				$data['total_group_activity'] = $wpdb->get_row($sql)->total;
			}

			echo PeepSoTemplate::exec_template('migrate', 'progress', $data);
		} else {
			echo PeepSoTemplate::exec_template('migrate', 'dashboard');
		}
	}

	public static function get_asset($ref) {
		$ret = plugin_dir_url(__FILE__) . 'assets/' . $ref;
		return ($ret);
	}

	public function migrate() {
		$migrate = new PeepsoMigrateAdminMigrate();
		switch ($_GET['action']) {
			case 'delete' :
				$migrate->deletePeepSoContent();
				break;
			case 'profile' :
				$migrate->startMigrateProfiles();
				break;
			case 'avatar' :
				$migrate->startMigrateAvatars();
				break;
			case 'friend' :
				$migrate->startMigrateFriends();
				break;
			case 'activity' :
				$migrate->startMigrateActivity();
				break;
			case 'notification' :
				$migrate->startMigrateNotifications();
				break;
			case 'message' :
				$migrate->startMigrateMessages();
				break;
			case 'profile_field' :
				$migrate->startMigrateProfileFields();
				break;
			case 'profile_field_value' :
				$migrate->startMigrateProfileFieldValue();
				break;
			case 'profile_field_user_data' :
				$migrate->startMigrateProfileFieldUserData();
				break;
			case 'group' :
				$migrate->startMigrateGroups();
				break;
			case 'group_member' :
				$migrate->startMigrateGroupMembers();
				break;
			case 'group_avatar' :
				$migrate->startMigrateGroupAvatars();
				break;
			case 'group_activity' :
				$migrate->startMigrateGroupActivity();
				break;
			case 'unpublish' :
				$migrate->unpublishBuddyPress();
				break;
		}
		die();
	}

	public static function url($action) {
		return admin_url('admin.php?page=pm-dashboard&action=' . $action);
	}

	public static function plugin_check($plugin) {

		$active_components = apply_filters('bp_active_components', bp_get_option('bp-active-components'));

		switch ($plugin) {
			case 'activity' :
				if (isset($active_components['activity']) && 1 == $active_components['activity']) {
					return TRUE;
				}
			case 'notification' :
				if (isset($active_components['notifications']) && 1 == $active_components['notifications']) {
					return TRUE;
				}
			case 'friend' :
				if (class_exists('PeepSoFriendsPlugin') && isset($active_components['friends']) && 1 == $active_components['friends']) {
					return TRUE;
				}
				break;
			case 'message' :
				if (class_exists('PeepSoMessagesPlugin') && isset($active_components['messages']) && 1 == $active_components['messages']) {
					return TRUE;
				}
				break;
			case 'profile_field' :
				if (isset($active_components['xprofile']) && 1 == $active_components['xprofile']) {
					return TRUE;
				}
				break;
			case 'group' :
				if (class_exists('PeepSoGroupsPlugin') && isset($active_components['groups']) && 1 == $active_components['groups']) {
					return TRUE;
				}
				break;
		}

		return FALSE;
	}

}

defined('WPINC') || die;
$peepso_migrate = new PeepsoMigrate();
?>
